<?php

class Telegram
{
    private string $endpoint = 'https://api.telegram.org/bot';
    private string $token;

    public function __construct(string $token)
    {
        $this->token = $token;
    }
    public function sendMessage(int $chatId, string $text, ?string $parseMode = null, ?array $entities = null, ?array $linkPreviewOptions = null, ?bool $disableNotification = null, ?bool $protectContent = null, ?bool $allowPaidBroadcast = null, ?string $messageEffectId = null, ?array $replyParameters = null, ?array $replyMarkup = null): array
    {
        $data = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => $parseMode,
            'entities' => $entities,
            'link_preview_options' => $linkPreviewOptions,
            'disable_notification' => $disableNotification,
            'protect_content' => $protectContent,
            'allow_paid_broadcast' => $allowPaidBroadcast,
            'message_effect_id' => $messageEffectId,
            'reply_parameters' => $replyParameters ? json_encode($replyParameters) : null,
            'reply_markup' => $replyMarkup ? json_encode($replyMarkup) : null,
        ];
        return $this->request('sendMessage', array_filter($data));
    }
    public function editMessageText(int|string $chatId, int $messageId, string $text, ?string $parseMode = null, ?array $entities = null, ?array $linkPreviewOptions = null, ?array $replyMarkup = null): array
    {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => $parseMode,
            'entities' => $entities,
            'link_preview_options' => $linkPreviewOptions,
            'reply_markup' => $replyMarkup ? json_encode($replyMarkup) : null,
        ];
        return $this->request('editMessageText', array_filter($data));
    }
    public function deleteMessage(int|string $chatId, int $messageId): array
    {
        return $this->request('deleteMessage', ['chat_id' => $chatId, 'message_id' => $messageId]);
    }
    public function unpinChatMessage(int|string $chatId, ?int $messageId = null): array
    {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
        ];
        return $this->request('unpinChatMessage', array_filter($data));
    }
    public function getChat(int|string $chatId): array
    {
        return $this->request('getChat', ['chat_id' => $chatId]);
    }
    // getChatAdministrators
    public function getChatAdministrators(int|string $chatId): array
    {
        return $this->request('getChatAdministrators', ['chat_id' => $chatId]);
    }
    public function canDelete(int $channelId, int $userId): bool
    {
        try {
            $admins = $this->getChatAdministrators($channelId);
        } catch (Throwable) {
            return false;
        }
        foreach ($admins['result'] as $admin) {
            if ($admin['user']['id'] == $userId) {
                return $admin['can_delete_messages'];
            }
        }
        return false;
    }
    public function sendMessageToMultipleUsers(array $userIds, string $text): void
    {
        foreach ($userIds as $userId) {
            try {
                $this->sendMessage($userId, $text);
            } catch (Exception $e) {
                error_log('Failed to send message to user ' . $userId . ': ' . $e->getMessage());
            }
        }
    }
    private function request(string $method, array $data = []): array
    {
        $url = $this->endpoint . $this->token . '/' . $method;
        $curl = curl_init($url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_POST, true);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        if ($httpCode !== 200) {
            throw new Exception('Telegram API request failed with code ' . $httpCode . ' - ' . $response);
        }
        $responseData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Failed to decode JSON response: ' . json_last_error_msg() . ' - ' . $response);
        }
        if (isset($responseData['ok']) && !$responseData['ok']) {
            throw new Exception('Telegram API error: ' . $responseData['description']);
        }
        return $responseData;
    }
}
